############################################################################
############## R script for BEST-WR analysis by S. Di Prima ################
############################################################################
# S. Di Prima et al., 2021. BEST-WR: An adapted algorithm for the hydraulic#
# characterization of hydrophilic and water-repellent soils. Journal of#####
# Hydrology 603, 126936. https://doi.org/10.1016/j.jhydrol.2021.126936 #####
############################################################################
####      ####        ###       #         #######  #########  ##      ######
####  ###   ##   ######   ##########   ##########  #########  ##  ###  #####
####  ###   ##   ######   ##########   ##########  #########  ##  ###  #####
####      ####     #####       #####   ###     ##  ###   ###  ##     #######
####  ###   ##   ############   ####   ###########  #  #  #  ###  ##   #####
####  ###   ##   ############   ####   ###########  #  #  #  ###  ####  ####
####       ###       ##        #####   ############   ###   ####  ####  ####
############################################################################

working_directory <- getwd()
if (!is.null(working_directory)) setwd(working_directory)

#install.packages("pracma")
library(pracma)


#########################  INPUT  #################################

path <- file.choose() 

input_name <- tools::file_path_sans_ext(basename(path))

input_table <- read.table(path, header = FALSE, sep = "", col.names = c("C1", "C2"))

ring_radius <- as.numeric(head(input_table$C2, n = 1))
theta_0 <- as.numeric(tail(head(input_table$C2, n = 2), n = 1))
theta_s <- as.numeric(tail(head(input_table$C2, n = 3), n = 1))

b <- 0.467
a <- 0.75/(ring_radius*(theta_s - theta_0))

N <- nrow(input_table) - 4

t_I = data.frame(t=tail(input_table$C1, n = N),
                 I=tail(input_table$C2, n = N)
                  )

t_I$t <- as.numeric(t_I$t)
t_I$I <- as.numeric(t_I$I)


################## Definition of the transient state ####################


last_3_points_t_I <- tail(t_I, n = 3)

regression <- lm(formula = last_3_points_t_I$I ~ last_3_points_t_I$t)

slope <- as.numeric(tail(coefficients(regression), n = 1))
int <- as.numeric(head(coefficients(regression), n = 1))

I_regr <- int + slope*t_I$t

E <- abs((t_I$I - I_regr)/t_I$I * 100)

t_I$test <- E < 2

t_I$N <- rev(rownames(t_I))

t_I$n <- ifelse(c(E < 2), "1", "0")

t_I$n <- as.numeric(t_I$n)


for(i in 1:N) {

  t_I$filter[i] <- paste0(sum(tail(t_I$n, n = N - i + 1)))

}

t_I$t_st <- ifelse(c(t_I$N == t_I$filter), t_I$t, "")
t_I$I_st <- ifelse(c(t_I$N == t_I$filter), t_I$I, "")

t_I$t_tr <- ifelse(c(t_I$N != t_I$filter), t_I$t, "")
t_I$I_tr <- ifelse(c(t_I$N != t_I$filter), t_I$I, "")


t_I$I_st <- as.numeric(t_I$I_st)
t_I$t_st <- as.numeric(t_I$t_st)

regression_st <- lm(formula = t_I$I_st ~ t_I$t_st)

slope_st <- as.numeric(tail(coefficients(regression_st), n = 1))
int_st <- as.numeric(head(coefficients(regression_st), n = 1))



#########################  Optimization  #################################

alpha <- 0
S <- 0

t_I$t_tr <- as.numeric(t_I$t_tr)
t_I$I_tr <- as.numeric(t_I$I_tr)
tk <- as.numeric(sum(table(t_I$t_tr))) #number of transient points
tk_tr=data.frame(t=head(t_I$t_tr, n = tk))

transient=data.frame(Iw=head(S*t_I$t^0.5+(a*(1-b)*S^2+b*slope_st)*t_I$t-S*(pi^0.5)/(2*(alpha)^0.5)*erf((alpha*t_I$t)^0.5)-(a*(1-b)*S^2+b*slope_st)/alpha*(1-exp(-alpha*t_I$t)), n = tk), 
                     Itr=head(t_I$I, n = tk))

tk_tr$t <- as.numeric(tk_tr$t)

min.RSS <- function(data, par) {
  with(data, sum(((par[2]*tk_tr$t^0.5+(a*(1-b)*par[2]^2+b*slope_st)*tk_tr$t-par[2]*(pi^0.5)/(2*(par[1])^0.5)*erf((par[1]*tk_tr$t)^0.5)-(a*(1-b)*par[2]^2+b*slope_st)/par[1]*(1-exp(-par[1]*tk_tr$t))) - Itr)^2))
}

starting_values <- expand.grid(
    alpha_start = c(0.001, 0.01, 0.1, 1, 10),
    S_start = c(0.001, 0.01, 0.1, 1, 10)
)

starting_values$alpha_start <- as.numeric(starting_values$alpha_start)
starting_values$S_start <- as.numeric(starting_values$S_start)

for(w in 1:25) {

alpha_start <- tail(head(starting_values$alpha_start, n = w), n =1)
S_start <- tail(head(starting_values$S_start, n = w), n =1)

optimization <- optim(par = c(alpha_start, S_start), fn = min.RSS, data = transient)

alpha_w <- optimization$par[1]
S_w <- optimization$par[2]

transient_opt=data.frame(Iw_opt=head(S_w*t_I$t^0.5+(a*(1-b)*S_w^2+b*slope_st)*t_I$t-S_w*(pi^0.5)/(2*(alpha_w)^0.5)*erf((alpha_w*t_I$t)^0.5)-(a*(1-b)*S_w^2+b*slope_st)/alpha_w*(1-exp(-alpha_w*t_I$t)), n = tk))

SSE_w <- sum((transient$Itr-transient_opt$Iw_opt)^2, na.rm=TRUE)


starting_values$alpha_opt[w] <- paste0(alpha_w, n = w)
starting_values$S_opt[w] <- paste0(S_w, n = w)
starting_values$SSE_w[w] <- paste0(SSE_w, n = w)

}

#########################  Selected values  ################################

starting_values$SSE_w <- as.numeric(starting_values$SSE_w)

SSE_w_min <- min(starting_values$SSE_w, na.rm = TRUE)

min_position <- which(starting_values$SSE_w == SSE_w_min) 

min_position <- as.numeric(min_position)

alpha_opt <- tail(head(starting_values$alpha_opt, n = min_position), n =1)
S_opt <- tail(head(starting_values$S_opt, n = min_position), n =1)

alpha_opt <- as.numeric(alpha_opt)
S_opt <- as.numeric(S_opt)

transient_opt=data.frame(Iw_opt=head(S_opt*t_I$t^0.5+(a*(1-b)*S_opt^2+b*slope_st)*t_I$t-S_opt*(pi^0.5)/(2*(alpha_opt)^0.5)*erf((alpha_opt*t_I$t)^0.5)-(a*(1-b)*S_opt^2+b*slope_st)/alpha_opt*(1-exp(-alpha_opt*t_I$t)), n = tk))

Ks = slope_st-a*S_opt^2

Er_fit <- 100*(SSE_w_min/sum(transient$Itr^2, na.rm=TRUE))^0.5

####################### Print Output ###############################

date <- format(Sys.time(), "%d.%b.%Y_%H.%M.%S")

filename <- paste0("Output_", input_name,"_", date, ".txt")

sink(filename)
cat("Ntot;",N,"\n")
cat("n_tr;",tk,"\n")
cat("slope [mm s^-1];",slope_st,"\n")
cat("int [mm];",int_st,"\n")
cat("S [mm s^-0.5];",S_opt,"\n")
cat("alpha [s^-1];",alpha_opt,"\n")
cat("Er [%];",Er_fit,"\n")
cat("SSE;",SSE_w_min,"\n")
cat("Ks [mm s^-1];",Ks,"\n")
sink()

########################### Plotting #############################

img_name <- paste0("Output_", input_name,"_", date, ".jpeg")

jpeg(img_name, width = 3200, height = 2400, res=600)

plot(t_I$t, t_I$I, pch = 21, col = "black", cex = 2, xlab="t [s]", ylab="I [mm]")
points(t_I$t_tr, t_I$I_tr, pch = 21, col = "red", cex = 2)
lines(tk_tr$t, transient_opt$Iw_opt, col = "blue", lwd=3)

dev.off()

##################################################################
