////////////////// Analytical quantification of cumulative infiltration and infiltration rate on the basis of Equations of Lassabatere et al. (2009)  //////////////////////////////////

////////////////// Scaled Cumulatine Infiltration //////////////////////////////////

///// I Direct formualtions /////

// Direct function t*(I*)

  function y = tI_star(I,u)                                     // to avoid problems of numerical computation of the function close to zero
     if I == 0 then y = 0; 
     else y = 1/(1-u)*(I-log((exp(u*I)+u-1)/u));
     end;
  endfunction
  
// Direct function q*(I*)

   function y = qI_star(I,u)                                    
      y = 1+u/(exp(u*I)-1);
  endfunction
    
// vectorialisation : to allow to calculate function for entries as vectors

  function y = tI_star_V(I,u)
      y = zeros(size(I,1),1);
      for i = 1:size(I,1),y(i) = tI_star(I(i),u);end;
  endfunction
  
  function y = qI_star_V(I,u)
      y = zeros(size(I,1),1);
      for i = 1:size(I,1),y(i) = qI_star(I(i),u);end;
  endfunction

  
///// II Approximations for formulations /////
  
// Short time approximation for cumulative infiltration

  function y = It_short_star(t)                                   
     y = (2*t)^0.5;
  endfunction
  
  function y = It_trans_star(t,u)                                   
     y = (2*t)^0.5+(2-u)/3*t;
  endfunction
  
  function y = It_sst_star(t,u)                                   
     y = t+1/(1-u)*log(1/u);
  endfunction
  
  
  
// Short time approximation for cumulative infiltration

  function y = qt_short_star(t)                                   
     y = (2*t)^(-0.5);
  endfunction
  
  function y = qt_trans_star(t,u)                                   
     y = (2*t)^(-0.5)+(2-u)/3;
  endfunction
  
  function y = qt_sst_star(t)                                   
     y = 1;
  endfunction

// vectorialisation : to allow to calculate function for entries as vectors

  function y = It_short_star_V(t)
      y = zeros(size(t,1),1);
      for i = 1:size(t,1),y(i) = It_short_star(t(i));end;
  endfunction

  function y = It_trans_star_V(t,u)
      y = zeros(size(t,1),1);
      for i = 1:size(t,1),y(i) = It_trans_star(t(i),u);end;
  endfunction
  
  function y = It_sst_star_V(t,u)
      y = zeros(size(t,1),1);
      for i = 1:size(t,1),y(i) = It_sst_star(t(i),u);end;
  endfunction
  

  function y = qt_short_star_V(t)
      y = zeros(size(t,1),1);
      for i = 1:size(t,1),y(i) = qt_short_star(t(i));end;
  endfunction

  function y = qt_trans_star_V(t,u)
      y = zeros(size(t,1),1);
      for i = 1:size(t,1),y(i) = qt_trans_star(t(i),u);end;
  endfunction
  
  function y = qt_sst_star_V(t)
      y = zeros(size(t,1),1);
      for i = 1:size(t,1),y(i) = qt_sst_star(t(i));end;
  endfunction
  
  // III validity times for short transient and steady states

  function y = t_short_star(eps)
     y = 1/(2*(1+eps)^2);
  endfunction
  
  function y = t_trans_star(u,eps)
     y = 1/(2*(1-((2-u)/3)+eps)^2);
  endfunction
  
  function y = t_sst_star(u,eps)
     y = 1/(1-u)*log((1+u/eps)^(1/u)/(1+1/eps));
  endfunction
  
// IV Implicit formulation I*(t*)

 function y = f2(I,m)
    y = t-(1/(1-u)*(I-log((exp(u*I)+u-1)/u)));                    // Equation: t - [right side of equation 7a] = 0
  endfunction
  
  // IV.a. Precise function depending I*(t*) with validity times defined in funciton to tolerance - eps fixed at 0.001
  
  function y = It_star(t,u);                                   // eps used here onkly to fix the limit between steady state and transient time
     if t < t_sst_star(u,10^-3) then I0 = It_trans_star(t,u)
         else I0 = It_sst_star(t,u);end;                                         // initial value for inversion based on equation for short times [eq 10b]
     [I1,v1,info] = lsqrsolve(I0,f2,1);                                           // solver of scilab I1 = root and V1 is error
     if info == 5 then y = It_sst_star(t,u);disp("Steady state used for t* ="+string(t));elseif t == 0 then y = 0; else y = I1;end;
  endfunction
 
  function y = It_star_V(t,u)                                   // same function for vectors
      y = zeros(size(t,1),1);
      for i = 1:size(t,1),y(i) = It_star(t(i),u);end;
  endfunction
  
//  // IV.a. Precise function depending I*(t*) with validity times defined in funciton to tolerance eps.
//  
//  function y = It_star_eps(t,u,eps);                                   // eps used here onkly to fix the limit between steady state and transient time
//     if t < t_sst_star(u,eps) then I0 = It_trans_star(t,u)
//         else I0 = It_sst_star(t,u);end;                                         // initial value for inversion based on equation for short times [eq 10b]
//     [I1,v1,info] = lsqrsolve(I0,f2,1);                                           // solver of scilab I1 = root and V1 is error
//     if info == 5 then y = It_sst_star(t,u);disp("Steady state used for t* ="+string(t));elseif t == 0 then y = 0; else y = I1;end;
//  endfunction
// 
//  function y = It_star_eps_V(t,u,eps)                                   // same function for vectors
//      y = zeros(size(t,1),1);
//      for i = 1:size(t,1),y(i) = It_star_eps(t(i),u,eps);end;
//  endfunction


//////////////////  Dimensional Cumulatine Infiltration //////////////////////////////////

// I. Fonctions gamma for scaling factors.

    function y = gamma_(Ks,K0,S)
      delta_K = Ks-K0;
      y(1,1) = S^2/(2*delta_K);
      y(2,1) = S^2/(2*delta_K^2);
    endfunction

// II. Functions for dimensional 1D infiltration

    // a Quasi-exact implicit formulation

    function y = I_1D(t,Ks,K0,S,u)
    
      z = gamma_(Ks,K0,S);                    // calculation of scaling factor
      gamma_I = z(1,1);
      gamma_t = z(2,1);
    
      t_star = t/gamma_t;                     // scaled infiltration for time
      I_star = It_star_V(t_star,u);           // use of I*(t*)
    
      y =  gamma_I*I_star+K0*t;               // dimensionalisation + K0*t
    
    endfunction
   
    // b Short time, transient and steady state pproximation

    function y = I_short_1D(t,Ks,K0,S) 

      z = gamma_(Ks,K0,S);                    // scaling factor
      gamma_I = z(1,1);
      gamma_t = z(2,1);
    
      t_star = t/gamma_t;                     // scaled infiltration
      I_star = It_short_star_V(t_star);       // use of IO(1)*(t*)
    
      y =  gamma_I*I_star+K0*t;               // dimensionalisation
    
    endfunction
   
    function y = I_trans_1D(t,Ks,K0,S,u) 
    
      z = gamma_(Ks,K0,S);                    // scaling factor
      gamma_I = z(1,1);
      gamma_t = z(2,1);

      t_star = t/gamma_t;                    // scaled infiltration
      I_star = It_trans_star_V(t_star,u);    // use of IO(2)*(t*)
    
      y =  gamma_I*I_star+K0*t;              // dimensionalisation
    
    endfunction
   
    function y = I_sst_1D(t,Ks,K0,S,u) 
   
      z = gamma_(Ks,K0,S);                   // scaling factor
      gamma_I = z(1,1);
      gamma_t = z(2,1);
    
      t_star = t/gamma_t;                   // scaled infiltration
      I_star = It_sst_star_V(t_star,u);     // use of IO(+inf)*(t*)
    
      y =  gamma_I*I_star+K0*t;             // dimensionalisation
    
    endfunction
   
// III. Functions for dimensional 3D infiltration

    // a Quasi-exact implicit formulation

    function y = I_3D(t,Ks,K0,S,rd,thetas,theta0,u,v)
    
      I1D = I_1D(t,Ks,K0,S,u);
      y =  I1D + v/(rd*(thetas-theta0))*S^2*t;               // dI13D = I1D + AS^2*t
    
     endfunction


   
    // I.b Short time, transient and steady state pproximation
    
    
    function y = I_short_3D(t,Ks,K0,S,rd,thetas,theta0,v)
    
    I1D = I_short_1D(t,Ks,K0,S);
    y =  I1D + v/(rd*(thetas-theta0))*S^2*t;               // dI13D = I1D + AS^2*t
    
    endfunction
  
   
    function y = I_trans_3D(t,Ks,K0,S,rd,thetas,theta0,u,v)
    
      I1D = I_trans_1D(t,Ks,K0,S,u);
      y =  I1D + v/(rd*(thetas-theta0))*S^2*t;               // dI13D = I1D + AS^2*t
    
    endfunction
   
    function y = I_sst_3D(t,Ks,K0,S,rd,thetas,theta0,u,v)
    
      I1D = I_sst_1D(t,Ks,K0,S,u);
      y =  I1D + v/(rd*(thetas-theta0))*S^2*t;               // dI13D = I1D + AS^2*t
    
    endfunction

//BEST WR

    function y =I_3D_WR(t,A,B,S,aWR,qexpinf,thetas,theta0);  
        y1 = S*t^0.5 
        y2=-S*%pi^0.5/(2*aWR^0.5)*erf((aWR*t)^0.5)
        y3=(A*(1-B)*S^2+B*qexpinf)*t 
        y4=-(A*(1-B)*S^2+B*qexpinf)*(1-exp(-aWR*t))/aWR   
        y=y1+y2+y3+y4  
    end
        
   
//BEST 3T-WR

    function y =I_3D_3T_WR(t,A,B,S,aWR,qexpinf,thetas,theta0);  
        y1 = S*t^0.5 
        y2=-S*%pi^0.5/(2*aWR^0.5)*erf((aWR*t)^0.5)
        y3=(A*(1-B)*S^2+B*qexpinf)*t 
        y4=-(A*(1-B)*S^2+B*qexpinf)*(1-exp(-aWR*t))/aWR   
        y5=0.76/9*(qexpinf-A*S^2)^2/S*t^1.5
        y6=-0.76/6*(qexpinf-A*S^2)^2/S
        y7=-t^0.5/aWR.*exp(-aWR*t)+%pi^0.5/2/aWR^1.5*erf((aWR*t)^0.5)   
        y=y1+y2+y3+y4+y5+y6*y7   
    end
        
   
